(function($, elementor) {
    'use strict';

    // Check if Elementor and our AI settings exist
    if (!elementor || !window.SpexoAiPageTranslator) {
        return;
    }

    // Translation progress tracking
    var translationState = {
        isTranslating: false,
        totalElements: 0,
        translatedElements: 0,
        failedElements: 0,
        currentElement: null,
        fromLang: '',
        toLang: '',
        isCancelled: false,
        currentRequests: [], // Store active AJAX requests to cancel them
        errorMessages: [] // Store error messages from failed translations
    };

    // Language options
    var languages = {
        'en': 'English',
        'es': 'Spanish (Español)',
        'fr': 'French (Français)', 
        'de': 'German (Deutsch)',
        'it': 'Italian (Italiano)',
        'pt': 'Portuguese (Português)',
        'ru': 'Russian (Русский)',
        'ja': 'Japanese (日本語)',
        'ko': 'Korean (한국어)',
        'zh': 'Chinese (中文)',
        'ar': 'Arabic (العربية)',
        'hi': 'Hindi (हिन्दी)',
        'nl': 'Dutch (Nederlands)',
        'pl': 'Polish (Polski)',
        'tr': 'Turkish (Türkçe)',
        'uk': 'Ukrainian (Українська)',
        'cs': 'Czech (Čeština)',
        'sv': 'Swedish (Svenska)',
        'no': 'Norwegian (Norsk)',
        'da': 'Danish (Dansk)',
        'fi': 'Finnish (Suomi)'
    };

    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        if (typeof text !== 'string') {
            text = String(text);
        }
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    /**
     * Shorten error message and convert URLs to links
     */
    function formatErrorMessage(error) {
        if (typeof error !== 'string') {
            error = String(error);
        }
        
        // Extract URL if present
        var urlMatch = error.match(/https?:\/\/[^\s]+/);
        var url = urlMatch ? urlMatch[0] : null;
        
        // Shorten the message (remove the long URL part and redundant text)
        var shortMessage = error;
        
        // Remove the URL from the message text
        if (url) {
            shortMessage = error.replace(url, '').trim();
            // Remove common trailing phrases
            shortMessage = shortMessage.replace(/\.?\s*For more information on this error,?\s*(read\s*)?(the\s*)?docs?:?\s*/i, '').trim();
            shortMessage = shortMessage.replace(/\.?\s*For more information,?\s*(read\s*)?(the\s*)?docs?:?\s*/i, '').trim();
            // Remove trailing periods if they're just before where URL was
            shortMessage = shortMessage.replace(/\.$/, '');
        }
        
        // If message is still too long, truncate it
        if (shortMessage.length > 150) {
            shortMessage = shortMessage.substring(0, 147) + '...';
        }
        
        // Build the formatted message with link
        var formatted = escapeHtml(shortMessage);
        if (url) {
            formatted += ' <a href="' + escapeHtml(url) + '" target="_blank" rel="noopener noreferrer" style="color: #0066cc; text-decoration: underline;">Learn more</a>';
        }
        
        return formatted;
    }

    /**
     * Check if premium version is active
     */
    function isPremiumActive() {
        // Check for premium indicators
        return !!(
            window.SpexoAddonsPro ||
            window.spexoAddonsPro ||
            (window.SpexoAiPageTranslator && window.SpexoAiPageTranslator.is_pro) ||
            (window.SpexoAiPageTranslator && window.SpexoAiPageTranslator.premium_active) ||
            document.querySelector('body.spexo-addons-pro') ||
            (typeof jQuery !== 'undefined' && jQuery('body').hasClass('spexo-addons-pro'))
        );
    }

    /**
     * Inject CSS styles for the translator
     */
    function injectTranslatorStyles() {
        if ($('#spexo-ai-translator-styles').length === 0) {
            const styles = `
                <style id="spexo-ai-translator-styles">
                    /* Translator Button Styles - Spexo Branding */
                    .spexo-ai-translator-btn {
                        background: linear-gradient(135deg, #6A1B9A, #E91E63) !important;
                        border: none !important;
                        color: white !important;
                        padding: 8px 12px !important;
                        border-radius: 6px !important;
                        font-size: 12px !important;
                        font-weight: 500 !important;
                        cursor: pointer !important;
                        display: inline-flex !important;
                        align-items: center !important;
                        gap: 6px !important;
                        transition: all 0.3s ease !important;
                        margin: 8px !important;
                        position: relative !important;
                        z-index: 10 !important;
                        text-decoration: none !important;
                        outline: none !important;
                        box-shadow: 0 2px 8px rgba(106, 27, 154, 0.3) !important;
                    }
                    .spexo-ai-translator-btn:hover {
                        background: linear-gradient(135deg, #7B1FA2, #D81B60) !important;
                        box-shadow: 0 4px 16px rgba(106, 27, 154, 0.4) !important;
                        transform: translateY(-1px) !important;
                    }
                    .spexo-ai-translator-btn:active {
                        transform: translateY(0) !important;
                        box-shadow: 0 2px 8px rgba(106, 27, 154, 0.3) !important;
                    }
                    .spexo-ai-translator-btn img {
                        width: 16px !important;
                        height: 16px !important;
                        flex-shrink: 0 !important;
                    }
                    .spexo-ai-translator-btn span {
                        white-space: nowrap !important;
                        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif !important;
                    }
                    
                    /* Location-specific styles */
                    
                    /* In panel header */
                    .spexo-translator-location-panel-header {
                        position: absolute !important;
                        top: 50% !important;
                        right: 16px !important;
                        transform: translateY(-50%) !important;
                        margin: 0 !important;
                        z-index: 1000 !important;
                    }
                    .spexo-translator-location-panel-header:hover {
                        transform: translateY(-50%) translateY(-1px) !important;
                    }
                    
                    /* In header within panel */
                    .spexo-translator-location-header-in-panel {
                        margin-left: auto !important;
                        margin-right: 8px !important;
                    }
                    
                    /* At top of panel */
                    .spexo-translator-location-panel-top {
                        width: calc(100% - 16px) !important;
                        margin: 8px !important;
                        justify-content: center !important;
                    }
                    
                    /* In general elementor panel */
                    .spexo-translator-location-elementor-panel {
                        margin: 8px !important;
                        align-self: flex-end !important;
                    }
                    
                    /* Toolbar button group integration styles */
                    .spexo-translator-location-left-group,
                    .spexo-translator-location-toolbar-stack,
                    .spexo-translator-location-toolbar,
                    .spexo-translator-location-grid-stack {
                        /* Material UI button styling is handled in the HTML structure */
                        display: inline-flex !important;
                    }
                    
                    /* Additional spacing for toolbar button */
                    .spexo-translator-location-left-group .spexo-ai-translator-btn,
                    .spexo-translator-location-toolbar-stack .spexo-ai-translator-btn,
                    .spexo-translator-location-grid-stack .spexo-ai-translator-btn {
                        margin-left: 8px !important;
                    }
                    
                    /* Compact popup styles */
                    .spexo-translator-popup.compact .spexo-translator-progress-text {
                        font-size: 14px;
                        margin-bottom: 8px;
                    }
                    
                    .spexo-translator-popup.compact .spexo-translator-current-element {
                        font-size: 12px;
                        margin-top: 8px;
                        color: #666;
                        overflow: hidden;
                        text-overflow: ellipsis;
                        white-space: nowrap;
                    }
                    
                    .spexo-translator-popup.compact .spexo-translator-stats {
                        margin: 12px 0;
                    }
                    
                    .spexo-translator-popup.compact .spexo-translator-stat {
                        margin: 0 8px;
                    }
                    
                    .spexo-translator-popup.compact .spexo-translator-stat-number {
                        font-size: 18px;
                    }
                    
                    .spexo-translator-popup.compact .spexo-translator-stat-label {
                        font-size: 11px;
                    }

                    /* Compact mode adjustments for custom fields */
                    .spexo-translator-popup.compact .spexo-prompt-examples {
                        padding: 6px;
                        margin-top: 4px;
                    }
                    
                    .spexo-translator-popup.compact .spexo-prompt-examples small {
                        font-size: 10px;
                    }
                    
                    .spexo-translator-popup.compact .spexo-pro-info {
                        font-size: 11px;
                        margin-top: 8px;
                        padding: 6px 8px;
                        background: #f8f9fa;
                        border-radius: 4px;
                        border-left: 3px solid #6A1B9A;
                    }

                    /* Element highlighting styles moved to preview iframe */

                    /* Ensure button appears properly in all panel locations */
                    #elementor-panel .spexo-ai-translator-btn,
                    .elementor-panel .spexo-ai-translator-btn {
                        max-width: 200px !important;
                        overflow: hidden !important;
                    }
                    
                    /* Responsive behavior */
                    @media (max-width: 600px) {
                        /* Hide text in panel buttons on small screens */
                        .spexo-ai-translator-btn span {
                            display: none !important;
                        }
                        .spexo-ai-translator-btn {
                            padding: 8px !important;
                            min-width: 32px !important;
                        }
                    }

                    /* Popup Overlay */
                    .spexo-translator-overlay {
                        position: fixed;
                        top: 0;
                        left: 0;
                        right: 0;
                        bottom: 0;
                        background: rgba(0,0,0,0.5);
                        z-index: 999999;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        transition: opacity 0.3s ease;
                    }
                    
                    .spexo-translator-overlay.hiding {
                        opacity: 0;
                        pointer-events: none;
                    }

                    /* Popup Container */
                    .spexo-translator-popup {
                        background: white;
                        padding: 24px;
                        border-radius: 8px;
                        box-shadow: 0 10px 40px rgba(0,0,0,0.3);
                        width: 90%;
                        max-width: 500px;
                        max-height: 80vh;
                        overflow-y: auto;
                        transition: all 0.3s ease;
                        transform: scale(1);
                    }
                    
                    /* Compact popup for top-right positioning */
                    .spexo-translator-popup.compact {
                        position: fixed;
                        top: 80px;
                        right: 20px;
                        width: 350px;
                        max-width: 350px;
                        padding: 16px;
                        z-index: 999999;
                        max-height: 400px;
                        transform: scale(1);
                        box-shadow: 0 8px 32px rgba(0,0,0,0.4);
                    }
                    
                    /* Compact popup header */
                    .spexo-translator-popup.compact h3 {
                        font-size: 16px;
                        margin: 0 0 12px 0;
                        display: flex;
                        justify-content: space-between;
                        align-items: center;
                    }
                    
                    /* Close button for compact popup */
                    .spexo-translator-close-btn {
                        background: none;
                        border: none;
                        font-size: 18px;
                        cursor: pointer;
                        color: #999;
                        width: 24px;
                        height: 24px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        border-radius: 3px;
                    }
                    
                    .spexo-translator-close-btn:hover {
                        background: #f0f0f0;
                        color: #333;
                    }
                    
                    /* Animation states */
                    .spexo-translator-popup.moving {
                        transition: all 0.5s cubic-bezier(0.25, 0.46, 0.45, 0.94);
                    }
                    
                    /* Notification banner animation */
                    @keyframes slideDown {
                        0% {
                            transform: translateY(-100%);
                            opacity: 0;
                        }
                        100% {
                            transform: translateY(0);
                            opacity: 1;
                        }
                    }
                    
                    /* Pulse animation for success numbers */
                    @keyframes pulse {
                        0% {
                            transform: scale(1);
                            opacity: 1;
                        }
                        50% {
                            transform: scale(1.1);
                            opacity: 0.8;
                        }
                        100% {
                            transform: scale(1);
                            opacity: 1;
                        }
                    }

                    .spexo-translator-popup h3 {
                        margin: 0 0 20px 0;
                        font-size: 18px;
                        color: #23282d;
                        display: flex;
                        align-items: center;
                        gap: 8px;
                    }

                    .spexo-translator-form {
                        display: flex;
                        flex-direction: column;
                        gap: 16px;
                    }

                    .spexo-translator-field {
                        display: flex;
                        flex-direction: column;
                        gap: 6px;
                    }

                    .spexo-translator-field label {
                        font-weight: 500;
                        color: #555;
                        font-size: 14px;
                    }

                    .spexo-translator-field select {
                        padding: 8px 12px;
                        border: 1px solid #ddd;
                        border-radius: 4px;
                        font-size: 14px;
                        height: auto;
                    }

                    .spexo-translator-field select:focus {
                        border-color: #6A1B9A;
                        box-shadow: 0 0 0 1px rgba(106, 27, 154, 0.3);
                        outline: none;
                    }

                    .spexo-translator-field input[type="text"] {
                        padding: 8px 12px;
                        border: 1px solid #ddd;
                        border-radius: 4px;
                        font-size: 14px;
                        margin-top: 6px;
                        transition: border-color 0.3s ease, box-shadow 0.3s ease;
                    }

                    .spexo-translator-field input[type="text"]:focus {
                        border-color: #6A1B9A;
                        box-shadow: 0 0 0 1px rgba(106, 27, 154, 0.3);
                        outline: none;
                    }

                    .spexo-custom-language-field {
                        margin-top: 8px;
                        display: none;
                        animation: slideDown 0.3s ease-out;
                    }

                    .spexo-custom-language-field.show {
                        display: block;
                    }

                    .spexo-custom-language-field input {
                        width: 100%;
                        box-sizing: border-box;
                    }

                    .spexo-custom-language-field label {
                        font-size: 13px;
                        color: #666;
                        margin-bottom: 4px;
                        display: block;
                    }

                    .spexo-pro-badge {
                        background: linear-gradient(135deg, #FFD700, #FFA500);
                        color: #333;
                        font-size: 10px;
                        font-weight: bold;
                        padding: 2px 6px;
                        border-radius: 3px;
                        margin-left: 6px;
                        vertical-align: middle;
                    }
                    
                    /* Style for disabled custom option when not premium */
                    .spexo-translator-field select option[value="custom"]:disabled {
                        color: #999;
                        background-color: #f5f5f5;
                    }
                    
                    /* Enhanced styling for custom language fields */
                    .spexo-custom-language-field.show input:focus {
                        border-color: #6A1B9A;
                        box-shadow: 0 0 0 2px rgba(106, 27, 154, 0.1);
                    }
                    
                    /* Info text for premium features */
                    .spexo-pro-info {
                        font-size: 13px;
                        color: #666;
                        margin-top: 4px;
                        font-style: italic;
                        line-height: 1.4;
                    }

                    .spexo-pro-info a {
                        color: #6A1B9A;
                        text-decoration: none;
                        font-weight: 500;
                    }

                    .spexo-pro-info a:hover {
                        color: #7B1FA2;
                        text-decoration: underline;
                    }

                    /* Prompt examples styling */
                    .spexo-prompt-examples {
                        margin-top: 6px;
                        padding: 8px;
                        background: #f8f9fa;
                        border-left: 3px solid #6A1B9A;
                        border-radius: 0 4px 4px 0;
                    }

                    .spexo-prompt-examples small {
                        color: #666;
                        font-size: 11px;
                        line-height: 1.4;
                        display: block;
                    }

                    @keyframes slideDown {
                        from {
                            opacity: 0;
                            max-height: 0;
                            transform: translateY(-10px);
                        }
                        to {
                            opacity: 1;
                            max-height: 100px;
                            transform: translateY(0);
                        }
                    }

                    /* Loading spinner animation */
                    @keyframes rotate {
                        from {
                            transform: rotate(0deg);
                        }
                        to {
                            transform: rotate(360deg);
                        }
                    }

                    /* Error popup specific styles - Spexo Branding */
                    .spexo-translator-btn-primary {
                        background: linear-gradient(135deg, #6A1B9A, #E91E63) !important;
                        color: white !important;
                        border: none !important;
                        padding: 12px 24px !important;
                        border-radius: 6px !important;
                        font-size: 14px !important;
                        font-weight: 500 !important;
                        cursor: pointer !important;
                        transition: all 0.3s ease !important;
                        text-decoration: none !important;
                        display: inline-flex !important;
                        align-items: center !important;
                        justify-content: center !important;
                        gap: 8px !important;
                        box-shadow: 0 2px 8px rgba(106, 27, 154, 0.3) !important;
                    }

                    .spexo-translator-btn-primary:hover {
                        background: linear-gradient(135deg, #7B1FA2, #D81B60) !important;
                        transform: translateY(-1px) !important;
                        box-shadow: 0 4px 16px rgba(106, 27, 154, 0.4) !important;
                    }

                    .spexo-translator-btn-secondary {
                        background: #f8f9fa !important;
                        color: #6c757d !important;
                        border: 1px solid #dee2e6 !important;
                        padding: 12px 24px !important;
                        border-radius: 6px !important;
                        font-size: 14px !important;
                        font-weight: 500 !important;
                        cursor: pointer !important;
                        transition: all 0.3s ease !important;
                        text-decoration: none !important;
                        display: inline-flex !important;
                        align-items: center !important;
                        justify-content: center !important;
                        gap: 8px !important;
                    }

                    .spexo-translator-btn-secondary:hover {
                        background: #e9ecef !important;
                        border-color: #adb5bd !important;
                        color: #495057 !important;
                    }

                    /* Progress bar styles */
                    .spexo-translator-progress {
                        text-align: center;
                        padding: 20px 0;
                    }

                    .spexo-translator-progress-text {
                        font-size: 16px;
                        font-weight: 500;
                        color: #333;
                        margin-bottom: 16px;
                    }

                    .spexo-translator-progress-bar {
                        width: 100%;
                        height: 8px;
                        background: #f0f0f0;
                        border-radius: 4px;
                        overflow: hidden;
                        margin-bottom: 12px;
                    }

                    .spexo-translator-progress-fill {
                        height: 100%;
                        background: linear-gradient(90deg, #6A1B9A, #E91E63);
                        border-radius: 4px;
                        transition: width 0.3s ease;
                        width: 0%;
                    }

                    .spexo-translator-current-element {
                        font-size: 14px;
                        color: #666;
                        margin-top: 8px;
                        font-style: italic;
                    }

                    .spexo-translator-stats {
                        display: flex;
                        justify-content: center;
                        gap: 20px;
                        margin: 20px 0;
                    }

                    .spexo-translator-stat {
                        text-align: center;
                    }

                    .spexo-translator-stat-number {
                        font-size: 24px;
                        font-weight: bold;
                        color: #6A1B9A;
                        display: block;
                    }

                    .spexo-translator-stat-label {
                        font-size: 12px;
                        color: #666;
                        margin-top: 4px;
                    }

                    .spexo-translator-actions {
                        display: flex;
                        gap: 12px;
                        justify-content: center;
                        margin-top: 20px;
                    }

                    /* Preview animation styles */
                    .spexo-translating-element {
                        position: relative !important;
                        overflow: hidden !important;
                    }

                    .spexo-translating-element::before {
                        content: '' !important;
                        position: absolute !important;
                        top: 0 !important;
                        left: -100% !important;
                        width: 100% !important;
                        height: 100% !important;
                        background: linear-gradient(90deg, transparent, rgba(91,3,255,0.3), transparent) !important;
                        animation: spexo-translate-sweep 2s infinite !important;
                        z-index: 1000 !important;
                        pointer-events: none !important;
                    }

                    .spexo-translated-element {
                        animation: spexo-translate-success 1.2s ease-out !important;
                    }

                    @keyframes spexo-translate-sweep {
                        0% {
                            left: -100%;
                        }
                        100% {
                            left: 100%;
                        }
                    }

                    @keyframes spexo-translate-success {
                        0% {
                            box-shadow: 0 0 0 0 rgba(76, 175, 80, 0.8);
                            transform: scale(1);
                        }
                        50% {
                            box-shadow: 0 0 0 10px rgba(76, 175, 80, 0.4);
                            transform: scale(1.02);
                        }
                        100% {
                            box-shadow: 0 0 0 0 rgba(76, 175, 80, 0), 
                                        0 0 20px rgba(76, 175, 80, 0.2);
                            transform: scale(1);
                        }
                    }
                    
                    /* Bounce animation for labels */
                    @keyframes spexo-translate-bounce {
                        0% { 
                            transform: translateX(-50%) translateY(-10px) scale(0.8);
                            opacity: 0;
                        }
                        50% { 
                            transform: translateX(-50%) translateY(-2px) scale(1.1);
                            opacity: 1;
                        }
                        70% { 
                            transform: translateX(-50%) translateY(-1px) scale(0.95);
                        }
                        100% { 
                            transform: translateX(-50%) translateY(0) scale(1);
                            opacity: 1;
                        }
                    }
                </style>
            `;
            $('head').append(styles);
        }
    }

    // Track button creation to prevent duplicates
    var buttonCreationInProgress = false;
    var lastButtonCreationTime = 0;
    
    /**
     * Add translator button to Elementor editor
     */
    function addTranslatorButton() {
        // Prevent rapid successive calls
        var now = Date.now();
        if (buttonCreationInProgress || (now - lastButtonCreationTime) < 1000) {
            return;
        }
        
        // Check if button already exists
        if ($('.spexo-ai-translator-btn').length > 0) {
            return;
        }
        
        buttonCreationInProgress = true;
        lastButtonCreationTime = now;
        
        // Try multiple locations for the button
        var locations = [
            { selector: '#elementor-editor-wrapper-v2 .MuiToolbar-root', location: 'mui-toolbar', priority: 1 },
            { selector: '#elementor-editor-wrapper-v2 .elementor-editor-panel-tools', location: 'v2-toolbar', priority: 2 },
            { selector: '.elementor-editor-panel-tools', location: 'toolbar', priority: 3 },
            { selector: '#elementor-panel-header', location: 'panel-header', priority: 4 },
            { selector: '.elementor-panel-header', location: 'header', priority: 5 }
        ];
        
        var buttonAdded = false;
        
        // Sort by priority
        locations.sort(function(a, b) { return a.priority - b.priority; });
        
        locations.forEach(function(loc) {
            var $target = $(loc.selector);
            if ($target.length && !$target.find('.spexo-ai-translator-btn').length && !buttonAdded) {
                // Double check this is NOT in the sidebar
                if (!$target.closest('.elementor-panel-navigation').length && 
                    !$target.closest('.elementor-panel-navigation-tabs').length &&
                    !$target.closest('#elementor-panel').length) {
                    
                    // Try to find the specific location between Elementor logo and Add Element button
                    var $logoButton = $target.find('[data-tooltip="Elementor"], .elementor-panel-header-menu-button, .MuiToolbar-root button:first-child');
                    var $addButton = $target.find('[data-tooltip="Add Element"], .elementor-panel-header-add-button, .MuiToolbar-root button:nth-child(2)');
                    
                    if ($logoButton.length && $addButton.length) {
                        // Insert between logo and add button
                        var $translatorBtn = createTranslatorButton(loc.location);
                        $addButton.before($translatorBtn);
                        buttonAdded = true;
                    } else {
                        // Fallback: add to the beginning of toolbar
                        var $translatorBtn = createTranslatorButton(loc.location);
                        $target.prepend($translatorBtn);
                        buttonAdded = true;
                    }
                }
            }
        });
        
        if (!buttonAdded) {
            console.log('Spexo AI Translator: Could not find suitable TOP TOOLBAR location for button injection');
        }
        
        // Reset flag after a delay
        setTimeout(function() {
            buttonCreationInProgress = false;
        }, 2000);
    }

    /**
     * Create translator button element
     */
    function createTranslatorButton(location) {
        // Try using the custom icon, fallback to the standard AI icon
        var iconUrl = SpexoAiPageTranslator.icon_url || SpexoAiPageTranslator.plugin_url + '/inc/modules/ai/assets/images/ai-generate.svg';
        
        var $translatorBtn = $('<button class="spexo-ai-translator-btn" title="Spexo AI Page Translator">' +
            '<img src="' + iconUrl + '" alt="Spexo AI Page Translator" onerror="this.src=\'' + iconUrl + '\'"/>' +
            '<span>Spexo AI Page Translator</span>' +
            '</button>');
        
        // Add location-specific class for different styling if needed
        $translatorBtn.addClass('spexo-translator-location-' + location);
        
        // Bind click event
        $translatorBtn.on('click', function(e) {
            e.preventDefault();
            // Check if button is disabled or translation is in progress
            if (translationState.isTranslating || $(e.currentTarget).prop('disabled')) {
                return;
            }
            showTranslatorPopup();
        });
        
        // Only log once per session to avoid spam
        if (!window.spexoTranslatorButtonLogged) {
            console.log('Spexo AI Translator button created for location:', location);
            window.spexoTranslatorButtonLogged = true;
        }
        return $translatorBtn;
    }

    /**
     * Show the translator popup
     */
    function showTranslatorPopup() {
        // Check API key first
        var $loadingOverlay = showLoadingOverlay();
        
        $.post(SpexoAiPageTranslator.ajax_url, {
            action: 'spexo_ai_text_check_limits',
            nonce: SpexoAiPageTranslator.generate_nonce
        }, function(response) {
            $loadingOverlay.remove();
            
            if (!response.success) {
                var errorMessage = response.data && response.data.message ? response.data.message : 'API key verification failed';
                showApiKeyError('Setup Required', errorMessage);
                return;
            }
            
            if (!response.data.api_key_valid) {
                var errorMessage = response.data.error_message || 'API key is missing or invalid';
                showApiKeyError('API Key Required', errorMessage);
                return;
            }
            
            createAndShowPopup();
        }).fail(function(xhr) {
            $loadingOverlay.remove();
            
            if (xhr.status === 0) {
                showApiKeyError('Connection Issue', 'Network connection failed. Please check your internet connection and try again.');
            } else {
                showApiKeyError('Temporary Issue', 'Server is temporarily unavailable. Please try again in a few minutes.');
            }
        });
    }

    /**
     * Show loading overlay
     */
    function showLoadingOverlay() {
        var $overlay = $('<div class="spexo-translator-overlay"></div>');
        var $popup = $('<div class="spexo-translator-popup" style="text-align: center; padding: 40px;"></div>');
        
        var loadingHtml = `
            <div style="margin-bottom: 16px;">
                <img src="${SpexoAiPageTranslator.plugin_url}inc/modules/ai/assets/images/ai-generate.svg" style="width:40px;height:40px;filter: invert(1); animation: rotate 1s linear infinite;"/>
            </div>
            <div style="font-size: 16px; color: #333; margin-bottom: 8px;">🔍 Verifying Setup...</div>
            <div style="font-size: 12px; color: #666;">Just checking that everything is ready for translation</div>
        `;
        
        $popup.html(loadingHtml);
        $overlay.append($popup);
        $('body').append($overlay);
        
        return $overlay;
    }

    /**
     * Show API key error with detailed information
     */
    function showApiKeyError(title, message) {
        // Aggressively remove any existing popups/overlays
        $('.spexo-translator-overlay').remove();
        $('.spexo-translator-popup').remove();
        
        // Wait a bit to ensure cleanup is complete
        setTimeout(function() {
            showApiKeyErrorDelayed(title, message);
        }, 100);
    }
    
    function showApiKeyErrorDelayed(title, message) {
        var settingsUrl = window.SpexoAiPageTranslator && window.SpexoAiPageTranslator.settings_url
            ? window.SpexoAiPageTranslator.settings_url
            : '/wp-admin/admin.php?page=spexo-ai-settings';
        
        var $overlay = $('<div class="spexo-translator-overlay"></div>');
        var $popup = $('<div class="spexo-translator-popup"></div>');
        
        var errorHtml = `
            <div style="text-align: center; margin-bottom: 32px;">
                <h3 style="margin: 0 0 16px 0; color: #2d3748; font-size: 20px; text-align: center;justify-content: center;">AI Translator Setup Required</h3>
                <p style="color: #718096; margin: 0; font-size: 14px;">This is completely safe and takes only 2 minutes!</p>
            </div>
            
            <div style="background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border: 1px solid #e2e8f0; padding: 20px; border-radius: 12px; margin-bottom: 20px;">
                <h4 style="color: #2d3748; margin: 0 0 12px 0; font-size: 16px;">📋 What you need to do:</h4>
                
                <div style="margin-bottom: 12px; display: flex; align-items: center;">
                    <span style="background: #48bb78; color: white; border-radius: 50%; width: 24px; height: 24px; display: flex; align-items: center; justify-content: center; margin-right: 12px; font-size: 12px; font-weight: bold;max-width: 24px;flex-basis: 46px;">1</span>
                    <span style="color: #4a5568;">Get an API key from <a href="https://platform.openai.com/api-keys" target="_blank" style="color: #5B03FF; text-decoration: none;line-height: 1.4;">OpenAI Platform</a> and top up your OpenAI account balance by at least $5</span>
                </div>
                
                <div style="margin-bottom: 12px; display: flex; align-items: center;">
                    <span style="background: #48bb78; color: white; border-radius: 50%; width: 24px; height: 24px; display: flex; align-items: center; justify-content: center; margin-right: 12px; font-size: 12px; font-weight: bold;max-width: 24px;flex-basis: 46px;">2</span>
                    <span style="color: #4a5568;">Go to <a href="${settingsUrl}" target="_blank" style="color: #5B03FF; text-decoration: none;line-height: 1.4;">AI Settings</a> and paste your API key</span>
                </div>
                
                <div style="margin-bottom: 0; display: flex; align-items: center;">
                    <span style="background: #48bb78; color: white; border-radius: 50%; width: 24px; height: 24px; display: flex; align-items: center; justify-content: center; margin-right: 12px; font-size: 12px; font-weight: bold;max-width: 24px;flex-basis: 46px;">3</span>
                    <span style="color: #4a5568;">Come back and click the Spexo AI Page Translator button again!</span>
                </div>
            </div>
            
            <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 16px; border-radius: 8px; margin-bottom: 20px;">
                <h4 style="color: #856404; margin: 0 0 8px 0; font-size: 14px;">💡 Why do I need this?</h4>
                <p style="color: #856404; margin: 0; font-size: 13px; line-height: 1.4;">
                    Spexo AI Page Translator uses OpenAI's GPT to translate your content. You only pay for what you use (usually $0.01-$0.10 per page).
                </p>
            </div>
            
            <div style="display: flex; gap: 12px;">
                <button class="spexo-translator-btn-secondary" id="spexo-api-key-close" style="flex: 1;">
                    I understand
                </button>
                <a href="${settingsUrl}" target="_blank" class="spexo-translator-btn-primary" style="flex: 1; text-decoration: none; display: flex; align-items: center; justify-content: center;">
                    Open AI Settings
                </a>
            </div>
        `;
        
        $popup.html(errorHtml);
        $overlay.append($popup);
        $('body').append($overlay);
        
        // Bind close event
        $('#spexo-api-key-close').on('click', function() {
            $overlay.remove();
        });
        
        // Close on overlay click
        $overlay.on('click', function(e) {
            if (e.target === $overlay[0]) {
                $overlay.remove();
            }
        });
    }

    /**
     * Create and show the main popup
     */
    function createAndShowPopup() {
        var $overlay = $('<div class="spexo-translator-overlay"></div>');
        var $popup = $('<div class="spexo-translator-popup"></div>');
        
        var isPro = isPremiumActive();
        var customOptionHtml = isPro ? 
            '<option value="custom">Custom language or prompt (PRO)</option>' : 
            '<option value="custom" disabled>Custom language or prompt (PRO)</option>';
        
        var upgradeUrl = 'https://spexo.com/pricing/?utm_source=ai-translator&utm_medium=plugin&utm_campaign=custom-prompts';
        var proInfoHtml = isPro ? 
            '<div class="spexo-pro-info" style="color: #4CAF50; border-left-color: #4CAF50;">✅ PRO Active: Use custom languages and translation prompts!</div>' :
            '<div class="spexo-pro-info">💎 <a href="' + upgradeUrl + '" target="_blank" style="color: #5B03FF; text-decoration: none;">Upgrade to PRO</a> to use custom languages, regional dialects and custom translation prompts (formal tone, technical style, etc.)!</div>';
        
        var popupContent = `
            <h3>
                <img src="${SpexoAiPageTranslator.plugin_url}inc/modules/ai/assets/images/ai-generate.svg" style="width:20px;height:20px;"/>
                Spexo AI Page Translator
            </h3>
            <div style="font-size: 12px; color: #666; margin-bottom: 16px; line-height: 1.4;">
                Translate to any language or transform text style (formal, casual, technical, etc.)
            </div>
            <div class="spexo-translator-form">
                <div class="spexo-translator-field">
                    <label>From Language</label>
                    <select id="spexo-from-lang">
                        <option value="auto">Auto-detect</option>
                        ${Object.keys(languages).map(code => 
                            `<option value="${code}">${languages[code]}</option>`
                        ).join('')}
                        ${customOptionHtml}
                    </select>
                    <div class="spexo-custom-language-field" id="spexo-custom-from-field">
                        <label>Custom language or translation prompt</label>
                        <input type="text" id="spexo-custom-from-lang" placeholder="e.g., Klingon, Old English, formal business tone, medical terminology..." />
                        <div class="spexo-prompt-examples">
                            <small>Examples: "Klingon", "Shakespeare English", "formal business style", "casual conversational tone"</small>
                        </div>
                    </div>
                </div>
                <div class="spexo-translator-field">
                    <label>To Language</label>
                    <select id="spexo-to-lang">
                        ${Object.keys(languages).map(code => 
                            `<option value="${code}" ${code === 'en' ? 'selected' : ''}>${languages[code]}</option>`
                        ).join('')}
                        ${customOptionHtml}
                    </select>
                    <div class="spexo-custom-language-field" id="spexo-custom-to-field">
                        <label>Custom language or translation style</label>
                        <input type="text" id="spexo-custom-to-lang" placeholder="e.g., Dothraki, Academic writing, pirate speak, baby talk..." />
                        <div class="spexo-prompt-examples">
                            <small>Examples: "Dothraki", "academic paper style", "pirate language", "simplified for children"</small>
                        </div>
                    </div>
                </div>
                ${proInfoHtml}
                <div class="spexo-translator-actions">
                    <button class="spexo-translator-btn-secondary" id="spexo-cancel-translation">
                        Cancel
                    </button>
                    <button class="spexo-translator-btn-primary" id="spexo-start-translation">
                        Start Translation
                    </button>
                </div>
            </div>
        `;
        
        $popup.html(popupContent);
        $overlay.append($popup);
        $('body').append($overlay);
        
        // Store references globally
        window.currentTranslatorPopup = $popup;
        window.currentTranslatorOverlay = $overlay;
        
        // Bind events for language selection
        bindLanguageSelectionEvents();
        
        // Bind events
        $('#spexo-cancel-translation').on('click', function() {
            if (!translationState.isTranslating) {
                $overlay.remove();
            }
        });
        
        $('#spexo-start-translation').on('click', function() {
            var result = getSelectedLanguages();
            
            if (!result.valid) {
                alert(result.error);
                return;
            }
            
            // Double-check API key before starting translation
            var $button = $(this);
            $button.prop('disabled', true).text('🔍 Verifying...');
            
            $.post(SpexoAiPageTranslator.ajax_url, {
                action: 'spexo_ai_text_check_limits',
                nonce: SpexoAiPageTranslator.generate_nonce
            }, function(response) {
                $button.prop('disabled', false).text('Start Translation');
                
                if (!response.success || !response.data.api_key_valid) {
                    var errorMessage = 'API key verification failed. Please check your API key in settings.';
                    if (response.data && response.data.error_message) {
                        errorMessage = response.data.error_message;
                    }
                    
                    showApiKeyError('Setup Required', errorMessage);
                    return;
                }
                
                // API key is valid, proceed with translation
                startTranslation(result.fromLang, result.toLang, $popup, $overlay);
                
            }).fail(function() {
                $button.prop('disabled', false).text('Start Translation');
                showApiKeyError('Connection Issue', 'Failed to connect right now. Please check your connection and try again.');
            });
        });
        
        // Close on overlay click only if not translating
        $overlay.on('click', function(e) {
            if (e.target === $overlay[0] && !translationState.isTranslating) {
                $overlay.remove();
            }
        });
    }

    /**
     * Bind events for language selection dropdowns
     */
    function bindLanguageSelectionEvents() {
        // Handle From Language selection
        $('#spexo-from-lang').on('change', function() {
            var selectedValue = $(this).val();
            var $customField = $('#spexo-custom-from-field');
            
            if (selectedValue === 'custom') {
                if (!isPremiumActive()) {
                    // Reset to previous value and show upgrade message
                    $(this).val('auto');
                    alert('Custom languages and translation prompts are a PRO feature. Please upgrade to Spexo Addons PRO to use custom languages or translation styles.');
                    return;
                }
                $customField.addClass('show');
                $('#spexo-custom-from-lang').focus();
            } else {
                $customField.removeClass('show');
            }
        });
        
        // Handle To Language selection
        $('#spexo-to-lang').on('change', function() {
            var selectedValue = $(this).val();
            var $customField = $('#spexo-custom-to-field');
            
            if (selectedValue === 'custom') {
                if (!isPremiumActive()) {
                    // Reset to previous value and show upgrade message
                    $(this).val('en');
                    alert('Custom languages and translation prompts are a PRO feature. Please upgrade to Spexo Addons PRO to use custom languages or translation styles.');
                    return;
                }
                $customField.addClass('show');
                $('#spexo-custom-to-lang').focus();
        } else {
                $customField.removeClass('show');
            }
        });
    }

    /**
     * Get selected languages with validation
     */
    function getSelectedLanguages() {
        var fromLang = $('#spexo-from-lang').val();
        var toLang = $('#spexo-to-lang').val();
        var customFromLang = $('#spexo-custom-from-lang').val().trim();
        var customToLang = $('#spexo-custom-to-lang').val().trim();
        
        // Handle custom from language
        if (fromLang === 'custom') {
            if (!customFromLang) {
                return {
                    valid: false,
                    error: 'Please enter a custom source language or translation prompt.'
                };
            }
            fromLang = customFromLang;
        }
        
        // Handle custom to language  
        if (toLang === 'custom') {
            if (!customToLang) {
                return {
                    valid: false,
                    error: 'Please enter a custom target language or translation style.'
                };
            }
            toLang = customToLang;
        }
        
        // Validate languages are different (except auto-detect)
        if (fromLang === toLang && fromLang !== 'auto') {
            return {
                valid: false,
                error: 'Source and target languages cannot be the same.'
            };
        }
        
        return {
            valid: true,
            fromLang: fromLang,
            toLang: toLang
        };
    }

    /**
     * Start the translation process
     */
    function startTranslation(fromLang, toLang, $popup, $overlay) {
        translationState.isTranslating = true;
        translationState.isCancelled = false;
        translationState.currentRequests = [];
        translationState.fromLang = fromLang;
        translationState.toLang = toLang;
        translationState.translatedElements = 0;
        translationState.failedElements = 0;
        translationState.errorMessages = [];
        
        // Get all elements to translate
        var elements = getAllElementsToTranslate();
        translationState.totalElements = elements.length;
        
        if (elements.length === 0) {
            translationState.isTranslating = false;
            alert('No translatable content found on this page.');
            return;
        }
        
        // Show progress
        showTranslationProgress($popup, $overlay);
        
        // Start translating elements
        translateElementsSequentially(elements, 0, $popup, $overlay);
    }

    /**
     * Get all elements that need translation
     */
    function getAllElementsToTranslate() {
        var elements = [];
        
        // Get the main document container using Elementor 3.0+ API (same as reference plugin)
        var documentContainer = elementor.documents.getCurrent().container;
        var elementorElements = [];
        
        // Use the new API to get children - for Elementor 3.0+
        if (documentContainer.children && typeof documentContainer.children.models !== 'undefined') {
            // Backbone collection - extract models
            elementorElements = documentContainer.children.models || [];
        } else if (documentContainer.elements && typeof documentContainer.elements.models !== 'undefined') {
            // Alternative property name in some Elementor versions
            elementorElements = documentContainer.elements.models || [];
        } else if (Array.isArray(documentContainer.children)) {
            // Fallback for older API
            elementorElements = documentContainer.children;
        } else {
            console.warn('Unable to find container children using any known API');
            elementorElements = [];
        }
        
        function processContainer(container) {
            var model = container.model;
            var elementType = model.get('elType');
            var widgetType = model.get('widgetType');
            
            // Process text-based widgets
            if (widgetType) {
                // Skip non-text widgets
                var nonTextWidgets = [
                    'spacer', 'divider', 'html', 'shortcode', 'sidebar',
                    'menu-anchor', 'read-more', 'google_maps', 'paypal_button',
                    'stripe_button', 'facebook_button', 'facebook_page',
                    'video', 'audio', 'iframe', 'code', 'wp-widget',
                    'map', 'rating', 'progress', 'counter', 'countdown',
                    'social-icons', 'share-buttons', 'login', 'lottie',
                    'image'  // Image widget should be skipped
                ];
                
                if (nonTextWidgets.indexOf(widgetType) !== -1) {
                    return; // Exit early for blacklisted widgets
                }
                
                var settings = model.get('settings');
                // Get attributes if it's a Backbone model
                var settingsAttributes = settings.attributes || settings || {};
                
                // Now check for text fields in remaining widgets
                var textFields = getTextFieldsForWidget(widgetType, settingsAttributes, container);
                
                // If we found text fields, process the widget
                if (textFields.length > 0) {
                    elements.push({
                        container: container,
                        widgetType: widgetType,
                        textFields: textFields,
                        elementId: model.get('id')
                    });
                    return;
                }
            }
            
            // Process child containers recursively using Elementor 3.0+ API
            if (container.children && container.children.length > 0) {
                // Check if children is a Backbone collection
                if (typeof container.children.models !== 'undefined') {
                    container.children.models.forEach(processContainer);
                } else if (Array.isArray(container.children)) {
                    container.children.forEach(processContainer);
                }
            }
        }
        
        elementorElements.forEach(processContainer);
        return elements;
    }

    /**
     * Get text fields for a specific widget type using Elementor control types (same as reference plugin)
     */
    function getTextFieldsForWidget(widgetType, settings, container) {
        var textFields = [];
        
        // Try to get widget controls schema from Elementor
        var controls = getWidgetControls(widgetType, container);
        
        if (controls && Object.keys(controls).length > 0) {
            // Look for text-based controls
            Object.keys(controls).forEach(function(controlName) {
                var control = controls[controlName];
                var controlType = control.type;
                var settingValue = settings[controlName];
                
                // Check if this is a text-based control type
                var textControlTypes = [
                    'text', 'textarea', 'wysiwyg', 'url', 'email', 
                    'password', 'search', 'tel', 'date', 'time', 
                    'datetime-local', 'month', 'week'
                ];
                
                if (textControlTypes.indexOf(controlType) !== -1) {
                    // Check if the field has a non-empty string value
                    if (settingValue && typeof settingValue === 'string' && settingValue.trim()) {
                        // Skip obviously non-translatable fields
                        var skipFields = [
                            '_element_id', '_css_classes', 'link', 'url', 'href', 
                            'custom_css', 'css_id', 'anchor', 'html_tag'
                        ];
                        
                        if (skipFields.indexOf(controlName) === -1) {
                            textFields.push({
                                field: controlName,
                                value: settingValue,
                                type: controlType === 'wysiwyg' ? 'wysiwyg' : 'text'
                            });
                        }
                    }
                }
                
                // Also check for repeater controls
                if (controlType === 'repeater' && settingValue) {
                    checkRepeaterFields(controlName, control, settingValue, textFields);
                }
            });
        } else {
            // Fallback: Use the original method for widgets without accessible controls
            var commonTextFields = [
                'title', 'text', 'content', 'description', 'subtitle', 'button_text',
                'heading_title', 'heading_subtitle', 'testimonial_content', 'testimonial_name',
                'title_text', 'description_text', 'content_text', 'editor'
            ];
            
            commonTextFields.forEach(function(field) {
                if (settings[field] && typeof settings[field] === 'string' && settings[field].trim()) {
                    textFields.push({
                        field: field,
                        value: settings[field],
                        type: field === 'editor' ? 'wysiwyg' : 'text'
                    });
                }
            });
        }
        
        return textFields;
    }
    
    /**
     * Get widget controls schema from Elementor (same as reference plugin)
     */
    function getWidgetControls(widgetType, container) {
        try {
            // Method 1: Try to get controls from container model
            if (container && container.model && container.model.get) {
                var model = container.model;
                
                // Try to get controls from the model's widget config
                if (model.config && model.config.controls) {
                    return model.config.controls;
                }
                
                // Try to get controls from the container settings
                if (container.settings && container.settings.controls) {
                    return container.settings.controls;
                }
            }
            
            // Method 2: Try to get controls from Elementor widgets registry
            if (window.elementor && elementor.widgets) {
                var widgetConfig = elementor.widgets.getWidgetType(widgetType);
                if (widgetConfig && widgetConfig.controls) {
                    return widgetConfig.controls;
                }
            }
            
            // Method 3: Try to get controls from elements manager
            if (window.elementor && elementor.elementsManager && container && container.model) {
                var elementView = elementor.elementsManager.getElementView(container.model.get('id'));
                if (elementView && elementView.model && elementView.model.controls) {
                    return elementView.model.controls;
                }
            }
            
            return null;
            
        } catch (error) {
            console.warn('Error getting widget controls:', error);
            return null;
        }
    }
    
    /**
     * Check repeater fields (basic implementation)
     */
    function checkRepeaterFields(repeaterName, repeaterControl, repeaterData, textFields) {
        try {
            // Get the fields schema for this repeater
            var repeaterFields = repeaterControl.fields || repeaterControl.controls || {};
            
            // Find text-based fields in the repeater schema
            var textFieldNames = [];
            Object.keys(repeaterFields).forEach(function(fieldName) {
                var fieldControl = repeaterFields[fieldName];
                var textControlTypes = ['text', 'textarea', 'wysiwyg', 'url', 'email'];
                
                if (textControlTypes.indexOf(fieldControl.type) !== -1) {
                    textFieldNames.push(fieldName);
                }
            });
            
            if (textFieldNames.length === 0) {
                return;
            }
            
            // Process repeater data
            if (repeaterData && typeof repeaterData === 'object' && repeaterData.models) {
                // Backbone collection
                var models = repeaterData.models || [];
                for (var i = 0; i < models.length; i++) {
                    var model = models[i];
                    var modelData = model.attributes || model.toJSON();
                    
                    for (var j = 0; j < textFieldNames.length; j++) {
                        var fieldName = textFieldNames[j];
                        if (modelData[fieldName] && typeof modelData[fieldName] === 'string' && modelData[fieldName].trim()) {
                            var fieldKey = repeaterName + '[' + i + '][' + fieldName + ']';
                            var fieldValue = modelData[fieldName];
                            
                            textFields.push({
                                field: fieldKey,
                                value: fieldValue,
                                type: 'text',
                                isRepeater: true,
                                repeaterKey: repeaterName,
                                repeaterIndex: i,
                                repeaterField: fieldName
                            });
                        }
                    }
                }
            } else if (Array.isArray(repeaterData)) {
                // Plain array
                for (var i = 0; i < repeaterData.length; i++) {
                    var item = repeaterData[i];
                    for (var j = 0; j < textFieldNames.length; j++) {
                        var fieldName = textFieldNames[j];
                        if (item[fieldName] && typeof item[fieldName] === 'string' && item[fieldName].trim()) {
                            var fieldKey = repeaterName + '[' + i + '][' + fieldName + ']';
                            textFields.push({
                                field: fieldKey,
                                value: item[fieldName],
                                type: 'text',
                                isRepeater: true,
                                repeaterKey: repeaterName,
                                repeaterIndex: i,
                                repeaterField: fieldName
                            });
                        }
                    }
                }
            }
        } catch (error) {
            console.warn('Error checking repeater fields:', error);
        }
    }

    /**
     * Show translation progress
     */
    function showTranslationProgress($popup, $overlay) {
        var progressHtml = `
            <div class="spexo-translator-progress">
                <div class="spexo-translator-progress-text">
                    🔄 Translating Page...
                </div>
                <div class="spexo-translator-progress-bar">
                    <div class="spexo-translator-progress-fill" id="spexo-progress-fill"></div>
                </div>
                <div class="spexo-translator-current-element" id="spexo-current-element">
                    Preparing translation...
                </div>
                <div class="spexo-translator-stats">
                    <div class="spexo-translator-stat">
                        <div class="spexo-translator-stat-number" id="spexo-total-elements">${translationState.totalElements}</div>
                        <div class="spexo-translator-stat-label">Total</div>
                    </div>
                    <div class="spexo-translator-stat">
                        <div class="spexo-translator-stat-number" id="spexo-translated-elements">0</div>
                        <div class="spexo-translator-stat-label">Translated</div>
                    </div>
                    <div class="spexo-translator-stat">
                        <div class="spexo-translator-stat-number" id="spexo-failed-elements">0</div>
                        <div class="spexo-translator-stat-label">Failed</div>
                    </div>
                </div>
                <div class="spexo-translator-actions">
                    <button class="spexo-translator-btn-secondary" id="spexo-cancel-translation-progress">
                        Cancel Translation
                    </button>
                </div>
            </div>
        `;
        
        $popup.find('.spexo-translator-form').html(progressHtml);
        
        // Bind cancel event
        $('#spexo-cancel-translation-progress').on('click', function() {
            translationState.isCancelled = true;
            translationState.isTranslating = false;
            
            // Cancel all active requests
            translationState.currentRequests.forEach(function(request) {
                if (request && request.abort) {
                    request.abort();
                }
            });
            translationState.currentRequests = [];
            
            $overlay.remove();
        });
    }

    /**
     * Translate elements sequentially
     */
    function translateElementsSequentially(elements, index, $popup, $overlay) {
        if (index >= elements.length || translationState.isCancelled) {
            // Translation complete
            showTranslationComplete($popup, $overlay);
            return;
        }
        
        var element = elements[index];
        var progress = ((index + 1) / elements.length) * 100;
        
        // Update progress
        $('#spexo-progress-fill').css('width', progress + '%');
        $('#spexo-current-element').text(`Translating ${element.widgetType}...`);
        
        // Translate this element
        translateElement(element, function(success) {
            if (success) {
                translationState.translatedElements++;
            } else {
                translationState.failedElements++;
            }
            
            // Update stats
            $('#spexo-translated-elements').text(translationState.translatedElements);
            $('#spexo-failed-elements').text(translationState.failedElements);
            
            // Continue with next element
            setTimeout(function() {
                translateElementsSequentially(elements, index + 1, $popup, $overlay);
            }, 500);
        });
    }

    /**
     * Translate a single element
     */
    function translateElement(element, callback) {
        var translatedFields = {};
        
        // Translate each text field
        var translatePromises = element.textFields.map(function(textField) {
            return new Promise(function(resolve) {
                translateText(textField.value, function(translatedText, success) {
                    if (success) {
                        translatedFields[textField.field] = translatedText;
                    }
                    resolve();
                });
            });
        });
        
        Promise.all(translatePromises).then(function() {
            // Update element settings
            updateElementSettings(element.container, translatedFields);
            callback(Object.keys(translatedFields).length > 0);
        });
    }

    /**
     * Translate text using AI
     */
    function translateText(text, callback) {
        // Check if translation was cancelled
        if (translationState.isCancelled) {
            callback(text, false);
            return;
        }
        
        var request = $.post(SpexoAiPageTranslator.ajax_url, {
            action: 'spexo_ai_translate_text',
                nonce: SpexoAiPageTranslator.translate_nonce,
                text: text,
            source_language: translationState.fromLang,
            target_language: translationState.toLang
        }, function(response) {
            // Remove request from active requests list
            var index = translationState.currentRequests.indexOf(request);
            if (index > -1) {
                translationState.currentRequests.splice(index, 1);
            }
            
            // Check if translation was cancelled while request was in progress
            if (translationState.isCancelled) {
                callback(text, false);
                return;
            }
            
            if (response.success && response.data && response.data.translated_text) {
                callback(response.data.translated_text, true);
            } else {
                // Capture error message from response
                var errorMessage = 'Unknown error occurred';
                if (response.data && response.data.message) {
                    errorMessage = response.data.message;
                } else if (response.message) {
                    errorMessage = response.message;
                } else if (typeof response === 'string') {
                    errorMessage = response;
                }
                
                // Store error message (avoid duplicates)
                if (translationState.errorMessages.indexOf(errorMessage) === -1) {
                    translationState.errorMessages.push(errorMessage);
                }
                
                callback(text, false);
            }
        }).fail(function(xhr, status, error) {
            // Remove request from active requests list
            var index = translationState.currentRequests.indexOf(request);
            if (index > -1) {
                translationState.currentRequests.splice(index, 1);
            }
            
            // Try to parse error from response
            var errorMessage = 'Network error occurred';
            try {
                if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMessage = xhr.responseJSON.data.message;
                } else if (xhr.responseText) {
                    var parsed = JSON.parse(xhr.responseText);
                    if (parsed.data && parsed.data.message) {
                        errorMessage = parsed.data.message;
                    } else if (parsed.message) {
                        errorMessage = parsed.message;
                    }
                }
            } catch (e) {
                // If parsing fails, use default message or status text
                if (xhr.statusText) {
                    errorMessage = 'Request failed: ' + xhr.statusText;
                }
            }
            
            // Store error message (avoid duplicates)
            if (translationState.errorMessages.indexOf(errorMessage) === -1) {
                translationState.errorMessages.push(errorMessage);
            }
            
            callback(text, false);
        });
        
        // Store the request so we can cancel it if needed
        translationState.currentRequests.push(request);
    }

    /**
     * Update element settings with translated text (using Elementor's command API like reference plugin)
     */
    function updateElementSettings(container, translatedFields) {
        try {
            if (!container || !container.model) {
                console.error('Container or model not found');
                return;
            }
            
            // Separate regular fields from repeater fields
            var regularFields = {};
            var repeaterUpdates = {};
            
            Object.keys(translatedFields).forEach(function(fieldKey) {
                var translatedValue = translatedFields[fieldKey];
                
                // Check if this is a repeater field
                var repeaterMatch = fieldKey.match(/^(.+)\[(\d+)\]\[(.+)\]$/);
                if (repeaterMatch) {
                    // This is a repeater field: repeaterKey[index][fieldName]
                    var repeaterKey = repeaterMatch[1];
                    var itemIndex = parseInt(repeaterMatch[2]);
                    var itemField = repeaterMatch[3];
                    
                    if (!repeaterUpdates[repeaterKey]) {
                        repeaterUpdates[repeaterKey] = {};
                    }
                    if (!repeaterUpdates[repeaterKey][itemIndex]) {
                        repeaterUpdates[repeaterKey][itemIndex] = {};
                    }
                    repeaterUpdates[repeaterKey][itemIndex][itemField] = translatedValue;
                } else {
                    // Regular field
                    regularFields[fieldKey] = translatedValue;
                }
            });
            
            // Apply regular field updates using Elementor's command API (triggers preview refresh)
            if (Object.keys(regularFields).length > 0) {
                if (window.$e && $e.run) {
                    // Use Elementor 3.0+ command API (same as reference plugin)
                    $e.run('document/elements/settings', {
                        container: container,
                        settings: regularFields
                    });
                } else {
                    // Fallback for older Elementor versions
                    var model = container.model;
                    var settings = model.get('settings');
                    Object.keys(regularFields).forEach(function(fieldKey) {
                        settings.set(fieldKey, regularFields[fieldKey]);
                    });
                    model.save({ status: 'autosave' });
                }
            }
            
            // Apply repeater field updates
            Object.keys(repeaterUpdates).forEach(function(repeaterKey) {
                var settings = container.model.get('settings');
                var currentSettings = settings.get(repeaterKey);
                
                // Handle Backbone Collections
                if (currentSettings && typeof currentSettings.models !== 'undefined') {
                    // Work with Backbone collection
                    Object.keys(repeaterUpdates[repeaterKey]).forEach(function(itemIndex) {
                        var index = parseInt(itemIndex);
                        if (currentSettings.models[index]) {
                            var model = currentSettings.models[index];
                            
                            // Update the specific fields in this repeater item
                            Object.keys(repeaterUpdates[repeaterKey][itemIndex]).forEach(function(fieldName) {
                                model.set(fieldName, repeaterUpdates[repeaterKey][itemIndex][fieldName]);
                            });
                        }
                    });
                    
                    // Use Elementor's command API to update the entire repeater
                    try {
                        var backboneData = currentSettings.toJSON ? currentSettings.toJSON() : 
                                          currentSettings.models.map(function(model) { 
                                              return model.toJSON ? model.toJSON() : model.attributes; 
                                          });
                        
                        var repeaterSettings = {};
                        repeaterSettings[repeaterKey] = backboneData;
                        
                        if (window.$e && $e.run) {
                            $e.run('document/elements/settings', {
                                container: container,
                                settings: repeaterSettings
                            });
                        } else {
                            settings.set(repeaterKey, backboneData);
                            container.model.save({ status: 'autosave' });
                        }
                    } catch (e) {
                        console.warn('Error updating repeater via Elementor API:', e);
                    }
                }
                // Handle regular arrays
                else if (Array.isArray(currentSettings)) {
                    var updatedRepeater = currentSettings.slice(); // Clone array
                    
                    Object.keys(repeaterUpdates[repeaterKey]).forEach(function(itemIndex) {
                        var index = parseInt(itemIndex);
                        if (updatedRepeater[index]) {
                            // Update the specific fields in this repeater item
                            Object.keys(repeaterUpdates[repeaterKey][itemIndex]).forEach(function(fieldName) {
                                updatedRepeater[index][fieldName] = repeaterUpdates[repeaterKey][itemIndex][fieldName];
                            });
                        }
                    });
                    
                    // Update the entire repeater field
                    var repeaterSettings = {};
                    repeaterSettings[repeaterKey] = updatedRepeater;
                    
                    if (window.$e && $e.run) {
                        $e.run('document/elements/settings', {
                            container: container,
                            settings: repeaterSettings
                        });
                    } else {
                        settings.set(repeaterKey, updatedRepeater);
                        container.model.save({ status: 'autosave' });
                    }
                }
            });
            
        } catch (error) {
            console.error('Error updating element settings:', error);
        }
    }

    /**
     * Show translation complete with stats
     */
    function showTranslationComplete($popup, $overlay) {
        // Build error messages HTML if there are errors
        var errorMessagesHtml = '';
        if (translationState.errorMessages.length > 0) {
            var uniqueErrors = translationState.errorMessages;
            errorMessagesHtml = `
                <div class="spexo-translator-errors" style="margin-top: 20px; padding: 15px; background: #FFF3CD; border-left: 4px solid #FF9800; border-radius: 4px;">
                    <div style="font-weight: bold; color: #856404; margin-bottom: 10px; display: flex; align-items: center; gap: 8px;">
                        <span style="font-size: 18px;">⚠️</span>
                        <span>Translation Errors:</span>
                    </div>
                    <div class="spexo-translator-error-list" style="margin-top: 10px;">
                        ${uniqueErrors.map(function(error, index) {
                            return `<div style="padding: 8px 0; color: #856404; font-size: 14px; line-height: 1.5;">
                                <strong>Error ${index + 1}:</strong> ${formatErrorMessage(error)}
                            </div>`;
                        }).join('')}
                    </div>
                    <div style="margin-top: 12px; padding-top: 12px; border-top: 1px solid rgba(133, 100, 4, 0.2);">
                        <div style="font-size: 13px; color: #856404;">
                            <strong>What to do:</strong>
                            <ul style="margin: 8px 0 0 20px; padding: 0;">
                                ${uniqueErrors.some(function(e) { return e.toLowerCase().includes('quota') || e.toLowerCase().includes('billing'); }) 
                                    ? '<li>Check your OpenAI account balance and billing details</li>' 
                                    : ''}
                                ${uniqueErrors.some(function(e) { return e.toLowerCase().includes('api key') || e.toLowerCase().includes('not configured'); }) 
                                    ? '<li>Verify your OpenAI API key is correctly configured in Spexo Settings</li>' 
                                    : ''}
                                ${uniqueErrors.some(function(e) { return e.toLowerCase().includes('token limit') || e.toLowerCase().includes('daily limit'); }) 
                                    ? '<li>Check your daily token limit in Spexo AI Settings</li>' 
                                    : ''}
                                <li>Try translating again after resolving the issue</li>
                                <li>If the problem persists, contact support</li>
                            </ul>
                        </div>
                    </div>
                </div>
            `;
        }
        
        var statsHtml = `
            <div class="spexo-translator-progress">
                <div class="spexo-translator-progress-text">
                    ${translationState.failedElements === translationState.totalElements ? '❌ Translation Failed' : translationState.failedElements > 0 ? '⚠️ Translation Partially Complete' : '✅ Translation Complete!'}
                </div>
                <div class="spexo-translator-stats">
                    <div class="spexo-translator-stat">
                        <div class="spexo-translator-stat-number" style="font-size: 32px; font-weight: bold; color: #2196F3;">${translationState.totalElements}</div>
                        <div class="spexo-translator-stat-label">Total Elements</div>
                    </div>
                    <div class="spexo-translator-stat">
                        <div class="spexo-translator-stat-number" style="font-size: 32px; font-weight: bold; color: #4CAF50; animation: ${translationState.translatedElements > 0 ? 'pulse 2s infinite' : 'none'};">
                            ${translationState.translatedElements}
                        </div>
                        <div class="spexo-translator-stat-label">✅ Translated</div>
                    </div>
                    <div class="spexo-translator-stat">
                        <div class="spexo-translator-stat-number" style="font-size: 32px; font-weight: bold; color: ${translationState.failedElements > 0 ? '#F44336' : '#999'};">${translationState.failedElements}</div>
                        <div class="spexo-translator-stat-label">${translationState.failedElements > 0 ? '❌' : '⚪'} Failed</div>
                    </div>
                </div>
                ${errorMessagesHtml}
                <div class="spexo-translator-actions" style="margin-top: 20px; text-align: center;">
                    <button class="spexo-translator-btn-primary" id="spexo-close-stats" style="background: #4CAF50; color: white; border: none; padding: 12px 30px; font-size: 16px; font-weight: bold; border-radius: 6px; cursor: pointer; box-shadow: 0 2px 4px rgba(0,0,0,0.2); min-width: 120px;">
                        Close Results
                    </button>
                </div>
            </div>
        `;
        
        $popup.find('.spexo-translator-form').html(statsHtml);
        
        $('#spexo-close-stats').on('click', function() {
            $overlay.remove();
        });
        
        // Reset translation state
        translationState.isTranslating = false;
        translationState.isCancelled = false;
        translationState.currentRequests = [];
    }

    /**
     * Wait for Elementor to be ready
     */
    function waitForElementor() {
        if (typeof elementor !== 'undefined' && elementor.hooks) {
            onElementorInit();
        } else {
            setTimeout(waitForElementor, 100);
        }
    }

    /**
     * Handle Elementor initialization
     */
    function onElementorInit() {
        // Check if AI Page Translator is enabled
        if (typeof SpexoAiPageTranslator !== 'undefined' && SpexoAiPageTranslator.translator_enabled === false) {
            console.log('Spexo AI Page Translator is disabled in settings');
            return;
        }
        
        // Inject styles first
        injectTranslatorStyles();
        
        // Add button immediately
        addTranslatorButton();

        // Also add button when panel opens
        if (typeof elementor !== 'undefined' && elementor.hooks) {
            elementor.hooks.addAction('panel/open_editor/widget', function() {
                setTimeout(addTranslatorButton, 100);
            });

            // Add button when navigator opens
            elementor.hooks.addAction('navigator/init', function() {
                setTimeout(addTranslatorButton, 100);
            });
        }

        // Monitor for panel changes
        observePanelChanges();
    }

    /**
     * Observe panel changes to re-add button if needed
     */
    function observePanelChanges() {
        var lastObserverCheck = 0;
        
        function createObserver() {
            return new MutationObserver(function(mutations) {
                var now = Date.now();
                // Throttle observer checks to prevent spam
                if (now - lastObserverCheck < 2000) {
                    return;
                }
                
                var shouldCheck = false;
                
                mutations.forEach(function(mutation) {
                    if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                        // Only check if button doesn't exist
                        if ($('.spexo-ai-translator-btn').length === 0) {
                            shouldCheck = true;
                        }
                    }
                });

                if (shouldCheck) {
                    lastObserverCheck = now;
                    setTimeout(addTranslatorButton, 500);
                }
            });
        }

        // Observe changes in the top toolbar (priority)
        var topToolbar = document.querySelector('#elementor-editor-wrapper-v2 .MuiToolbar-root');
        if (topToolbar) {
            var topObserver = createObserver();
            topObserver.observe(topToolbar, {
                childList: true,
                subtree: true
            });
            console.log('Observing changes in top toolbar');
        }
        
        // Also observe the main editor wrapper for structural changes
        var editorWrapper = document.querySelector('#elementor-editor-wrapper-v2');
        if (editorWrapper) {
            var wrapperObserver = createObserver();
            wrapperObserver.observe(editorWrapper, {
                childList: true,
                subtree: false
            });
            console.log('Observing changes in editor wrapper');
        }

        // Observe changes in the main panel (fallback)
        var panel = document.querySelector('#elementor-panel');
        if (panel) {
            var panelObserver = createObserver();
            panelObserver.observe(panel, {
                childList: true,
                subtree: true
            });
            console.log('Observing changes in elementor panel');
        }

        // Observe the main Elementor editor area
        var editorArea = document.querySelector('#elementor-editor-wrapper, .elementor-editor-wrapper');
        if (editorArea) {
            var editorObserver = createObserver();
            editorObserver.observe(editorArea, {
                childList: true,
                subtree: true
            });
            console.log('Observing changes in editor area');
        }
    }

    /**
     * Initialize the translator
     */
    function initTranslator() {
        // Wait for Elementor to be fully loaded
        $(window).on('elementor:init', function() {
            console.log('Elementor initialized, setting up AI Translator');
            // Add small delay to ensure Material UI is rendered
            setTimeout(onElementorInit, 500);
        });

        // Fallback if elementor:init doesn't fire
        setTimeout(function() {
            console.log('Fallback initialization for AI Translator');
            onElementorInit();
        }, 3000);
        
        // Additional fallback for when Material UI components are ready
        setTimeout(function() {
            if (!document.querySelector('.spexo-ai-translator-btn')) {
                console.log('Material UI fallback initialization for AI Translator');
                onElementorInit();
            }
        }, 5000);
    }

    // Initialize when DOM is ready
    $(document).ready(function() {
        console.log('DOM ready, initializing AI Translator');
        initTranslator();
        
        // Global event handler for close buttons (backup protection)
        $(document).off('click.aiTranslatorGlobal').on('click.aiTranslatorGlobal', '.spexo-translator-close-btn', function(e) {
            console.log('🚫 Global close button clicked');
            if (translationState.isTranslating) {
                console.log('🛑 Stopping translation via global handler');
                translationState.isCancelled = true;
                translationState.isTranslating = false;
                
                // Cancel all active requests
                translationState.currentRequests.forEach(function(request) {
                    if (request && request.abort) {
                        request.abort();
                    }
                });
                translationState.currentRequests = [];
                
                // Show cancellation notice
                var $notice = $('<div style="position: fixed; top: 120px; right: 20px; background: #ff9800; color: white; padding: 8px 12px; border-radius: 4px; font-size: 14px; z-index: 1000000;">Translation cancelled</div>');
                $('body').append($notice);
        setTimeout(function() {
                    $notice.fadeOut(300, function() {
                        $notice.remove();
                    });
        }, 2000);
    }

            // Close popup/overlay
            var $popup = $(this).closest('.spexo-translator-popup');
            if ($popup.hasClass('compact')) {
                $popup.remove();
            } else {
                $popup.closest('.spexo-translator-overlay').remove();
            }
            
            // Reset state and re-enable button
            translationState.isTranslating = false;
            translationState.isCancelled = false; 
            translationState.currentRequests = [];
            
            e.preventDefault();
            e.stopPropagation();
        });
    });

})(jQuery, window.elementor);