<?php
if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * @class    CCF7_Brevo_Integration_Class
 * @category Class
 * @author   Nikhil Tiwari
 */
class CCF7_Brevo_Integration_Class
{
    protected static $_instance = null;
    public $glSettings = array();

    // Get singleton instance
    public static function get_instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    // Constructor
    public function __construct()
    {
        $this->hooks();
    }

    public function hooks()
    {
        // Enqueue necessary styles and scripts for admin pages
        add_action('admin_enqueue_scripts', array($this, 'ccf7b_enqueue_styles_scripts'));

        // Add a custom Brevo tab to the CF7 form editor
        add_filter('wpcf7_editor_panels', array($this, 'ccf7b_add_brevo_tab'));

        // Save Brevo settings when a CF7 form is saved
        add_action('wpcf7_save_contact_form', array($this, 'ccf7b_save_brevo_settings'));

        // Handle the submission of CF7 forms to Brevo
        add_action('wpcf7_mail_sent', array($this, 'ccf7b_cf7_to_brevo'));

        // Register AJAX action hooks for logged-in and non-logged-in users to verify the Brevo API key
        add_action('wp_ajax_verify_brevo_api_key', array($this, 'ccf7b_verify_brevo_api_key_callback'));
        add_action('wp_ajax_verify_brevo_api_key', array($this, 'ccf7b_verify_brevo_api_key_callback'));
    }

    /**
     * Callback function to verify the Brevo API key.
     * This function will be called via AJAX when a request is made.
     */
    public function ccf7b_verify_brevo_api_key_callback()
    {
        // Verify the AJAX nonce for security
        check_ajax_referer('save_brevo_settings', 'nonce');

        // Retrieve the API key and form ID from the AJAX request   
        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;

        // Verify the API key using the Brevo API
        $is_verified = $this->ccf7b_verify_brevo_api_key($api_key);

        // Check if the API key was successfully verified
        if ($is_verified['success']) {
            // Update the post meta with the verified API key
            update_post_meta($post_id, '_brevo_api_key', $api_key);
            // Send a successful JSON response back to the AJAX request
            wp_send_json_success($is_verified['message']);
        } else {
            // The API key verification failed
            wp_send_json_error($is_verified['message']);
        }
    }

    /**
     * Send CF7 form data to Brevo when a form is submitted.
     *
     * @param object $contact_form The CF7 form object that was submitted.
     */
    public function ccf7b_cf7_to_brevo($contact_form)
    {
        // Get the ID of the submitted form
        $form_id = $contact_form->id();

        // Get the current submission instance
        $submission = WPCF7_Submission::get_instance();

        // Check if a submission exists
        if ($submission) {
            // Get posted form data
            $form_data  = $submission->get_posted_data();

            // Retrieve Brevo API key from post meta
            $api_key = get_post_meta($form_id, '_brevo_api_key', true);

            // Prepare the form data to be sent to Brevo
            $fields_to_send = $this->ccf7b_prepare_brevo_fields($form_id, $form_data, $api_key);

            // Send the data to Brevo
            $result = $this->ccf7b_send_to_brevo($api_key, $fields_to_send);

            // Handle result if needed
            if ($result['status'] === 'error') {
                // Log error or take other action
                error_log($result['message']);
            }
        }
    }

    /**
     * Send prepared fields to Brevo.
     *
     * @param string $api_key The Brevo API key.
     * @param array $fields The prepared fields to send.
     * @return array Result of the send operation.
     */
    public function ccf7b_send_to_brevo($api_key, $fields)
    {

        $result =  $this->ccf7b_send_cf7_data_to_brevo($api_key, $fields);
        return $result;
    }

    /**
     * Send CF7 data to Brevo API.
     *
     * @param string $api_key The Brevo API key.
     * @param array $fields The fields to send.
     * @param int|null $list_id Optional group ID for specific Brevo list.
     * @return array The result of the API request.
     */
    public function ccf7b_send_cf7_data_to_brevo($api_key, $fields, $list_id = null)
    {
        // Define the Brevo API endpoint for creating a contact
        $url = 'https://api.brevo.com/v3/contacts';
        $list_ids = $this->ccf7b_brevo_api_response('lists', $api_key);
        $listId = array_column($list_ids['data'], 'id');

        // Prepare the payload for the API request
        $data = $this->ccf7b_prepare_brevo_payload($fields, $listId);

        // Make the API request
        return $this->ccf7b_execute_brevo_request($url, $api_key, $data);
    }

    /**
     * Execute the Brevo API request to add a contact.
     *
     * @param string $url The Brevo API URL.
     * @param string $api_key The API key for authenticating the request.
     * @param array $data The payload to send with the request.
     * @return array The response from the Brevo API.
     */
    private function ccf7b_execute_brevo_request($url, $api_key, $data)
    {
        // Make the API request
        $response = wp_remote_post($url, [
            'headers' => [
                'Content-Type' => 'application/json',
                'api-key' => $api_key,
            ],
            'body' => wp_json_encode($data),
        ]);

        // Check if the response is a WP_Error (network or HTTP errors)
        if (is_wp_error($response)) {
            return array(
                'status' => 'error',
                'message' => 'Request failed: ' . $response->get_error_message(),
            );
        }


        // Retrieve the response body
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);


        // Check the status code for success
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 201) { // Brevo returns 201 for successful creation
            return array(
                'status' => 'success',
                'message' => 'Contact added successfully.',
                'data' => $data,
            );
        } else {
            return array(
                'status' => 'error',
                'message' => isset($data['message']) ? $data['message'] : 'An error occurred.',
                'data' => $data,
            );
        }
    }

    /**
     * Prepare the payload for the Brevo API request.
     *
     * @param array $fields The fields containing the contact's data.
     * @return array The prepared payload for the API request.
     */
    private function ccf7b_prepare_brevo_payload($fields, $listId)
    {
        return [
            'email' => $fields['email'], // Use email address from fields
            'attributes' => isset($fields['attributes']) ? $fields['attributes'] : [], // Additional merge fields
            'listIds' => isset($listId) ? $listId : [],
            'updateEnabled' => true,
        ];
    }

    /**
     * Prepare form data for submission to Brevo.
     *
     * @param int $post_id The CF7 form ID.
     * @param array $form_data The submitted form data.
     * @param string $api_key The Brevo API key.
     * @return array The prepared data to send to Brevo.
     */
    public function ccf7b_prepare_brevo_fields($post_id, $form_data, $api_key)
    {
        // Retrieve field mappings from post meta
        $brevo_field_mappings = get_post_meta($post_id, '_brevo_field_mappings', true) ?: [];

        // Fetch brevo fields
        $brevo_fields = $this->ccf7b_brevo_api_response_get_field('attributes', $api_key);

        $fields = [];

        // Iterate through the Brevo fields
        foreach ($brevo_fields as $brevo_field) {
            $this->ccf7b_process_brevo_field($brevo_field, $brevo_field_mappings, $form_data, $fields);
        }

        // Return the prepared fields
        return $fields;
    }

    /**
     * Retrieves and filters fields from the Brevo API response.
     *
     * @param string $endpoint The API endpoint from which to fetch fields.
     * @param string $api_key The API key used for authentication with the Brevo API.
     * @return array An array of filtered fields, including the email field.
     */
    public function ccf7b_brevo_api_response_get_field($endpoint, $api_key)
    {
        $brevo_response = $this->ccf7b_brevo_api_response($endpoint, $api_key);
        $brevo_all_fields = isset($brevo_response['data']) ? $brevo_response['data'] : []; // Ensure we only get fields
        $exclude_fields = array('BLACKLIST', 'READERS', 'CLICKERS', 'EXT_ID', 'SMS', 'CONTACT_TIMEZONE');

        // Filter out unwanted fields
        $filtered_fields = array_filter($brevo_all_fields, function ($field) use ($exclude_fields) {
            return !in_array($field['name'], $exclude_fields);
        });

        // Re-index the array
        $brevo_fields = array_values($filtered_fields);

        // Manually add the email field at the beginning
        $email_field = array(
            'name' => 'EMAIL',
            'category' => 'normal',
            'type' => 'text',
            'field_key' => 'email'
        );

        array_unshift($brevo_fields, $email_field);
        return $brevo_fields;
    }

    /**
     * Process a single Brevo field and map it to CF7 form data.
     *
     * @param array $brevo_field The Brevo field to process.
     * @param array $mappings The field mappings between CF7 and Brevo.
     * @param array $form_data The form data submitted by the CF7 form.
     * @param array &$fields The array to hold prepared fields for Brevo.
     */
    private function ccf7b_process_brevo_field($brevo_field, $mappings, $form_data, &$fields)
    {
        $field_key = isset($brevo_field['field_key']) ? $brevo_field['field_key'] : strtolower($brevo_field['name']);
        // Check if the brevo tag exists in the mappings
        if (array_key_exists($field_key, $mappings)) {
            // Get the CF7 field mapping based on the brevo tag
            $cf7_field_key = $mappings[$field_key];
            $cf7_field_clean_key = preg_replace('/\[.*?\s|\]/', '', $cf7_field_key);

            // Check if the form data has the cleaned field key
            if (isset($form_data[$cf7_field_clean_key])) {
                $field_value = $form_data[$cf7_field_clean_key];

                // Handle array values (e.g., checkboxes or multiple selections)
                if (is_array($field_value)) {
                    // Convert array to comma-separated string
                    $field_value = implode(', ', $field_value);
                }

                // Assign values dynamically to brevo fields
                $this->ccf7b_assign_field_value($brevo_field, $field_value, $form_data, $fields);
            }
        }
    }

    /**
     * Assign field values to the appropriate Brevo fields based on the Brevo tag.
     *
     * @param array $brevo_field The Brevo field being processed.
     * @param mixed $field_value The value from the CF7 form associated with the Brevo field.
     * @param array $form_data The form data submitted by the CF7 form.
     * @param array &$fields The array to hold prepared fields for Brevo.
     */
    private function ccf7b_assign_field_value($brevo_field, $field_value, $form_data, &$fields)
    {
        $field_key = isset($brevo_field['field_key']) ? $brevo_field['field_key'] : strtolower($brevo_field['name']);
        if ($brevo_field['name'] === 'EMAIL') {
            $fields[$field_key] = $field_value; // Directly assign email
        } else {
            // For all other fields, assign them under attributes
            $fields['attributes'][$field_key] = $field_value;
        }
    }

    /**
     * Add the Brevo settings tab to the CF7 editor.
     *
     * @param array $panels Existing editor panels.
     * @return array Modified panels with Brevo tab.
     */
    public function ccf7b_add_brevo_tab($panels)
    {
        // Define a new panel for Brevo settings
        $panels['brevo-integration-panel'] = [
            'title'    => __('Brevo Integration', 'centous-integration-for-contact-form-7-and-brevo'),
            'callback' => [$this, 'ccf7b_render_brevo_panel_content'],
        ];

        // Create a new array to maintain the desired order
        $new_order = [];

        $current_index = 0;

        foreach ($panels as $key => $panel) {
            if ($current_index === 4) {
                // Add Brevo tab before the current tab if we reached the fourth position
                $new_order['brevo-integration-panel'] = $panels['brevo-integration-panel'];
            }
            $new_order[$key] = $panel;
            $current_index++;
        }

        // If we have fewer than 4 tabs, just append it at the end
        if ($current_index < 4) {
            $new_order['brevo-integration-panel'] = $panels['brevo-integration-panel'];
        }

        return $new_order;
    }

    /**
     * Save Brevo settings when a CF7 form is saved.
     *
     * This function verifies the nonce for security, retrieves the API key,
     * and if valid, saves the field mappings submitted via the CF7 form settings.
     *
     * @param ContactForm $contact_form The CF7 form instance being processed.
     */
    public function ccf7b_save_brevo_settings($contact_form)
    {
        // Verify nonce for security to prevent CSRF attacks
        if (!isset($_POST['brevo_settings_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['brevo_settings_nonce'])), 'save_brevo_settings')) {
            return; // Invalid nonce
        }


        // Get the ID of the current form
        $post_id = $contact_form->id();

        // Get the ID of the current form
        $api_key = get_post_meta($post_id, '_brevo_api_key', true);

        // Verify the Brevo API key before proceeding to save settings
        if ($this->ccf7b_verify_brevo_api_key($api_key)) {

            // Check if field mappings are submitted
            if (isset($_POST['brevo_settings'])) {
                // Sanitize and save field mappings
                $mappings = array_map('sanitize_text_field', wp_unslash($_POST['brevo_settings']));

                // Update the post meta with the sanitized mappings
                $updated_fields = update_post_meta($post_id, '_brevo_field_mappings', $mappings);

                //add params on url 
                if ($updated_fields) {
                    $cf7_version = get_plugin_data(WP_PLUGIN_DIR . '/contact-form-7/wp-contact-form-7.php')['Version'];
                    if ($cf7_version >= 6.0) {
                        $redirect_url = admin_url('admin.php?page=wpcf7&post=' . $post_id . '&action=edit&active-tab=brevo-integration-panel&section=field-mapping');
                        wp_redirect($redirect_url);
                        exit;
                    } else {
                        $redirect_url = admin_url('admin.php?page=wpcf7&post=' . $post_id . '&action=edit&active-tab=4&section=field-mapping');
                        wp_redirect($redirect_url);
                        exit;
                    }
                }
            }
        }
    }

    /**
     * Retrieves the response from a specific Brevo API endpoint using the provided API key.
     *
     * @param string $endpoint The API endpoint to call (e.g., 'subscribers').
     * @param string $api_key The Brevo API key to use for authentication.
     * @return array Returns the decoded JSON response as an associative array, or an empty array on error.
     */
    public function ccf7b_brevo_api_response($endpoint, $api_key)
    {
        // Brevo API URL for retrieving contact attributes
        $url = 'https://api.brevo.com/v3/contacts/' . $endpoint . '';

        // Set up the arguments for the request, including the API key in headers
        $args = array(
            'headers' => array(
                'api-key' => $api_key,
                'Accept' => 'application/json',
            ),
            'timeout' => 15, // Timeout after 15 seconds
        );

        // Make the API request using wp_remote_get
        $response = wp_remote_get($url, $args);

        // Check for errors in the request
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => 'Request failed: ' . $response->get_error_message());
        }

        // Get the response body and decode the JSON
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        // Check if the response contains valid attributes
        if (isset($data[$endpoint]) && is_array($data[$endpoint])) {

            return array('success' => true, 'data' => $data[$endpoint]);
        } else {
            return array('success' => false, 'message' => 'Unable to retrieve fields or unauthorized request.');
        }
    }

    /**
     * Retrieves all shortcodes (fields) used in a Contact Form 7 form by its ID.
     *
     * @param int $post_id The ID of the Contact Form 7 post.
     * @return array An array of shortcodes representing the form fields.
     */
    public function ccf7b_get_contact_from7_fields($post_id)
    {
        // Get the current form instance
        $form = WPCF7_ContactForm::get_instance($post_id);
        // Get the form content (HTML with shortcodes)
        $form_content = $form->prop('form');

        $cf7_shortcodes = [];

        // Regular expression to match all shortcodes (e.g., [text* your-name])
        if (preg_match_all('/\[(\w+)(\*?)\s+([^\]]+)\]/', $form_content, $matches, PREG_SET_ORDER)) {

            foreach ($matches as $shortcode) {
                $field_type = $shortcode[1]; // e.g., email, text, select, checkbox
                $is_required = $shortcode[2]; // If the field is required (denoted by '*')
                $field_details = $shortcode[3]; // Get the rest of the shortcode content

                // Split field details by space and take the first element as the field name
                $field_name = explode(' ', $field_details)[0]; // e.g., your-email, your-subject

                // Store the shortcode in an array
                $cf7_shortcodes[] = '[' . esc_attr($field_type . $is_required . ' ' . $field_name) . ']'; // e.g., [email* your-email]
            }
        }
        return $cf7_shortcodes;
    }

    /**
     * Verifies the Brevo API key by making a request to the Brevo API.
     *
     * @param string $api_key The API key to be verified.
     * @return bool Returns true if the API key is valid, false otherwise.
     */
    public function ccf7b_verify_brevo_api_key($api_key)
    {
        // Brevo API URL for account information
        $url = 'https://api.brevo.com/v3/account';

        // Set up the arguments for the request, including the API key in headers
        $args = array(
            'headers' => array(
                'api-key' => $api_key,
                'Accept' => 'application/json',
            ),
            'timeout' => 15, // Timeout after 15 seconds
        );

        // Make the API request using wp_remote_get
        $response = wp_remote_get($url, $args);

        // Check for errors in the request
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => 'Request failed: ' . $response->get_error_message());
        }
        // Get the response body and decode the JSON
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        // Check if the response contains valid account information
        if (isset($data['email'])) {
            return array('success' => true, 'message' => 'API key is valid. Account email: ' . $data['email']);
        } else {
            return array('success' => false, 'message' => 'Invalid API key or unauthorized request.');
        }
    }

    /**
     * Renders the field mapping section in the form editor.
     *
     * @param WP_Post $post The current post object.
     */
    public function ccf7b_field_mapping_section($post)
    {
        // Prepare data to be passed to the template
        $data = array(
            'post' => $post
        );
        // Load the field mapping settings template
        helper_ccf7b_object()->ccf7b_get_template('ccf7b-field-mapping-page.php', 'cf7-brevo-integration-module', $data, false);
    }

    public function ccf7b_render_brevo_panel_content($post)
    {
        $data = array(
            'post' => $post
        );
        helper_ccf7b_object()->ccf7b_get_template('ccf7b-api-key.php', 'cf7-brevo-integration-module', $data, false);
    }

    /**
     * Renders the Brevo settings section in the form editor.
     *
     * @param WP_Post $post The current post object.
     */
    public function ccf7b_setting_section_brevo($post)
    {
        // Prepare data to be passed to the template
        $data = array(
            'post' => $post
        );
        // Load the Brevo settings template
        helper_ccf7b_object()->ccf7b_get_template('ccf7b-api-setting-tabs.php', 'cf7-brevo-integration-module', $data, false);
    }

    /**
     * Renders the documentation section for the Mailchimp integration.
     * This function is responsible for preparing data and loading the corresponding template.
     */
    public function ccf7b_documentation_section()
    {
        // Prepare data to be passed to the template
        $data = array();

        // Load the field mapping settings template
        helper_ccf7b_object()->ccf7b_get_template('ccf7b-documentation-page.php', 'cf7-brevo-integration-module', $data, false);
    }

    public function ccf7b_enqueue_styles_scripts()
    {
        // Check user capabilities to ensure they have access
        if (!current_user_can('manage_options')) {
            return;
        }
        // Register the style
        wp_register_style('cf7_brevo_style', CCF7B_URL . 'modules/cf7-brevo-integration-module/css/style.css', array(), CCF7B_VERSION, 'all');
        wp_enqueue_style('cf7_brevo_style');

        // Register the script
        wp_register_script('cf7_brevo_script', CCF7B_URL . 'modules/cf7-brevo-integration-module/js/script.js', array('jquery'), CCF7B_VERSION, true);
        wp_enqueue_script('cf7_brevo_script');

        // Pass Ajax URL to the JavaScript file
        wp_localize_script('cf7_brevo_script', 'wep_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('save_brevo_settings'),
            'cf7_version' => get_plugin_data(WP_PLUGIN_DIR . '/contact-form-7/wp-contact-form-7.php')['Version']
        ));
    }
}
// Function to get the singleton instance
function class_ccf7_brevo_Integration_object()
{
    return CCF7_Brevo_Integration_Class::get_instance();
}
// Initialize the class
class_ccf7_brevo_Integration_object();
